/*
	File:		NewtonMemory.h

	Contains:	Memory interface for Newton build system

	Copyright:	 1992-1995 by Apple Computer, Inc.  All rights reserved.

	Derived from v15 internal.

*/

#ifndef __NEWTONMEMORY_H
#define __NEWTONMEMORY_H

#ifndef __NEWTONTYPES_H
#include "NewtonTypes.h"
#endif

//----------------------------------------------------------------------
//		Newton PDA-specific stuff
//----------------------------------------------------------------------

typedef long Size;

// memory debugging types, from KernelTypes.h
typedef unsigned char HeapBlockType;

/*
**	Opaque memory manager types
**
*/
typedef void *	Heap;



#ifdef __cplusplus
	#define DEFAULT_NIL = 0
#else
	#define	DEFAULT_NIL
#endif

#ifdef __cplusplus
extern "C" {
#endif


/*	----------------------------------------------------------------
**
**	General operations
**
*/
NewtonErr		MemError(void);						/* per-task result of last memory mgr call */


/*	----------------------------------------------------------------
**
**	"Ptr" operations
**
**		Note that 'SetPtrSize' does not exist -- it is gone from the
**		Macintosh suite (use ReallocPtr instead, which can move the Ptr
**		block and is MUCH more effective than the Mac's SetPtrSize).
**
*/

Ptr				NewPtr(Size size);
Ptr				NewNamedPtr(Size size, ULong name);
Ptr				NewPtrClear(Size byteCount);
void			DisposPtr(Ptr p);
Size			GetPtrSize(Ptr p);
Ptr				ReallocPtr(Ptr, Size);			/* like realloc */
Heap			PtrToHeap(Ptr);


/*	----------------------------------------------------------------
**
**		"Locking" a Ptr has the following semantics:
**
**			o	The Ptr's guts are accessible to interrupt code.  (Since
**				VM-heaps are possibly backed by compression or some actual
**				storage device, it is unsafe to touch memory from interrupt
**				code without locking it);
**
**				Note that handles cannot be locked.
**
**			o	The Ptr is NOT gauranteed to be physically contiguous (it
**				may cross a page boundary).  It is unsafe to DMA into or
**				out of it.  Use NewWiredPtr if you want to use DMA;
**
**			o	The Ptr is cacheable (or not), as determined by the heap
**				creation parameters;
**
**			o	If you lock a Ptr, you are responsible for unlocking it
**				before you free it;
**
*/
NewtonErr		LockPtr(Ptr);					/* lock down in virtual memory */
NewtonErr		UnlockPtr(Ptr);


/*	----------------------------------------------------------------
**
**	Wired Ptrs
**
**		These are for use by DMA; a wired pointer block is g'teed to
**		be contiguous in RAM, and will have a fixed physical address
**		for its lifetime.
**
**		Limitations are:
**
**			o	Size limit of a smidgeon less than 4K (assume 32 bytes
**				less than 4K and you won't be disappointed);
**
**			o	The operations are limited to allocate/deallocate in
**				order to keep the implementation flexible (e.g. you
**				must keep track of size of wired ptrs yourself);
**
**		Use wired pointers with care; they gum up the allocation of
**		pages.
**
*/
Ptr				NewWiredPtr(Size size);
void			DisposeWiredPtr(Ptr);


/*	----------------------------------------------------------------
**
**	Handle operations
**
**		Note that HLock returns a pointer (which you should use).
**
*/
Handle			NewHandle(Size size);
Handle			NewNamedHandle(Size size, ULong name);
Handle			NewHandleClear(Size byteCount);
void			DisposHandle(Handle h);
Size			GetHandleSize(Handle h);
NewtonErr		SetHandleSize(Handle h, Size size);
void *			HLock(Handle h);
void			HUnlock(Handle h);
void			HSetState(Handle h, char savedCount);			/* deprecated */
char			HGetState(Handle h);							/* deprecated */
NewtonErr		HandToHand(Handle *hPtr);
Handle			CopyHandle(Handle);
void			MoveHHi(Handle);								/* NOP for now */
Heap			HandleToHeap(Handle);


/*	----------------------------------------------------------------
**
**	Fake handles (e.g. handles to things in ROM)
**
**		DisposeHandle is allowed (destroys the handle);
**
**		GetHandleSize returns specified size
**
**		SetHandleSize returns -1
**
**		HLock is a NOP (returns address)
**
**		HUnlock, HSetState are NOPs
**
**		HandToHand, CopyHandle will result in correct copies
**
*/
Handle			NewFakeHandle(void * address, Size size);
Boolean			IsFakeHandle(Handle);


/*	----------------------------------------------------------------
**
**	Tagging and naming heap blocks
**
**		Names and tags are mutually exclusive.  If you assign a name
**		to a block in a shared heap, that block will not be automatically
**		freed when the owner task dies.
**
*/
TObjectId		GetPtrOwner(Ptr);
void			SetPtrOwner(Ptr, TObjectId);
TObjectId		GetHandleOwner(Handle);
void			SetHandleOwner(Handle, TObjectId);


/*	----------------------------------------------------------------
**
**	Naming heap blocks
**
*/
ULong			GetPtrName(Ptr);
void			SetPtrName(Ptr, ULong);
ULong			GetHandleName(Handle);
void			SetHandleName(Handle, ULong);


/*	----------------------------------------------------------------
**
**	Typing heap blocks
**
*/
HeapBlockType	GetPtrType(Ptr);
void			SetPtrType(Ptr, HeapBlockType);
HeapBlockType	GetHandleType(Handle);
void			SetHandleType(Handle, HeapBlockType);


/*	----------------------------------------------------------------
**
**	Heap Operations
**
*/
Heap			GetHeap(void);				/* get the task's current heap */
void			SetHeap(Heap);				/* set the task's current heap */

NewtonErr		NewSegregatedVMHeap(	TObjectId	defaultDomain,		/* domain for heap (or zero for current env's default) */
										Size		ptrSize,			/* max size of Ptr allocation */
										Size		handleSize,			/* max size of handle allocation */
										Heap *		pResult,			/* out: new heap */
										ULong		options );			/* various options (see below) */

NewtonErr		NewVMHeap(				TObjectId	defaultDomain,		/* domain for heap (or zero for current env's default) */
										Size		maxSize,			/* max size of both Ptr and handle allocation */
										Heap *		pResult,			/* out: new heap */
										ULong		options);			/* various options (see below) */

NewtonErr 		NewPersistentVMHeap(	TObjectId	domainId,			// domain (zero ==> default for caller's environment)
										Size		maxSize,			// max storage (not TOO huge, since there's static overhead)
										Heap *		pResult,			// out: the heap
										ULong		options,			// various options
										ULong		name);				// name for persistent heap

NewtonErr 		DeletePersistentVMHeap(	ULong		name);				// name for persistent heap


/*	----------------------------------------------------------------
**
**	Options for heap creation
**
*/
enum
{
	 kHeapNotCacheable					= 0x00000001		// don't cache heap
	,kPersistentHeap					= 0x40000000		// heap is recoverable
};


NewtonErr		NewHeapAt(				VAddr,							/* where heap is in VM */
										Size,							/* size of heap */
										Heap * pResult);				/* out: new heap */

NewtonErr		ResurrectVMHeap(Heap oldHeap);							/* bring a heap at some vaddr back to life after a reboot */
NewtonErr		DestroyVMHeap(Heap DEFAULT_NIL);						/* clobber a heap */
NewtonErr		ShrinkHeapLeaving(Heap, Size amountLeftFree);			/* attempt to shrink a heap */
NewtonErr		AddSemaphoreToHeap(Heap DEFAULT_NIL);					/* attach a semaphore to a heap */
NewtonErr		ClobberHeapSemaphore(Heap DEFAULT_NIL);					/* clobber a heap's semaphore (careful!) */
NewtonErr		ZapHeap(Heap, ULong verification, Boolean isPersistent);/* clobber the specified heap */
#define			kZapHeapVerification	'-><-'

Size			TotalFreeInHeap(Heap DEFAULT_NIL);
Size			LargestFreeInHeap(Heap DEFAULT_NIL);
unsigned long	CountFreeBlocks(Heap DEFAULT_NIL);
Size			TotalUsedInHeap(Heap DEFAULT_NIL);
Size			MaxHeapSize(Heap DEFAULT_NIL);
Size			GetHeapReleaseable(Heap DEFAULT_NIL);


/*	----------------------------------------------------------------
**
**	System memory availability information
**
**		TotalSystemFree
**
**			The number of bytes currently available for allocation.  This does
**			not include internal (e.g. 1-4K) fragmentation in heaps and stacks.
**
**		MaxPossibleSystemFree
**
**			An empirically determined number that is the maximum number of bytes
**			available for allocation before the user has stored anything.
**
**		SystemRAMSize
**
**			e.g. 512K for original MessagePad
**
*/
Size			TotalSystemFree(void);
Size			SystemRAMSize(void);


/*	----------------------------------------------------------------
**
**	Lightweight heap checking
**
**	 (This will always be available in the product -- it doesn't cost
**	  any RAM, and is fairly small codewise, and its utility is high...).
**
*/
NewtonErr CheckHeap
	(
		Heap		opaque_heap,		/* heap to check (nil for current heap) */
		void **		whereSmashed		/* out: if non-nil and heap is smashed, address of accident */
	);


/*	----------------------------------------------------------------
**
**	Memory operations
**
**		If you are in doubt as to what to call to shuffle large amounts
**		of memory around, use BlockMove (which is actually the amazing
**		ARM super-optimized memmove).  DON'T invent your own memory-moving
**		routines.
**
*/
void BlockMove(const void * srcPtr, void * destPtr, Size byteCount);
int  EqualBytes(const void *, const void *, Size);
void FillBytes(void *, Size length, UChar pattern);
void FillLongs(void *, Size length, ULong pattern);
void ZeroBytes(void *, Size length);
void XORBytes(const void * src1, const void * src2, void * dest, Size size);


/*
**	Task-safe heap walking
**
**		Seed-based.  Ask the heap for a seed, then pass the seed in on
**		every operation.  If the seed changes out from under you, get
**		another seed and start over.
**
**		Returns block type.
**
**		Returns 'kMM_HeapEndBlock' at the end of the heap.
**
*/
long		HeapSeed(Heap);
int			NextHeapBlock
				(	Heap			opaque_heap,		/* heap to walk	*/
					long			seed,				/* permission to walk heap	*/
					void *			fromBlock,			/* from block, nil for first block	*/
					void **			pFoundBlock,		/* out, next block */
					void ***		pFoundBlockHandle,	/* out, block's handle, if any */
					int *			pFoundBlockType,	/* block's type (see below)	*/
					char *			pFoundBlockTag,		/* block's tag */
					Size *			pFoundBlockSize,	/* block's size (apparent size, not true size)	*/
					TObjectId *		pFoundBlockOwner	/* block's owner	*/
				);

/*
**	Block types returned by NextHeapBlock
**
*/
enum
{
	 kMM_HeapSeedFailure		/* wups -- try again!	*/
	,kMM_InternalBlock			/* internal block */
	,kMM_HeapHeaderBlock		/* heap header	*/
	,kMM_HeapPtrBlock			/* Ptr block	*/
	,kMM_HeapHandleBlock		/* Handle block (result of NextHeapBlock is the handle)	*/
	,kMM_HeapEndBlock			/* end-of-heap	*/
	,kMM_HeapMPBlock			/* master-pointer block	*/
	,kMM_HeapFreeBlock			/* free block	*/
};


/*	----------------------------------------------------------------
**
**	SetMemMgrBreak  --  Set a memory-manager break condition
**
**		This is a little bit like a logic analyzer hooked up to the
**		memory manager.  When one of the conditions you set up here
**		triggers, a DebugStr indicating the condition is hit.
**
**		SetMemMgrBreak( 0 );					// reset all breaks
**		SetMemMgrBreak(	'tag!' );				// begin tagging allocated blocks with call-chain hashes
**		SetMemMgrBreak( 'tag?', hash_value);	// break on allocate when call-chain hash matches
**		SetMemMgrBreak( '<', size);				// break on requests/resizes less than 'size'
**		SetMemMgrBreak( '>=', size);			// break on requests/resizes >= 'size'
**		SetMemMgrBreak( '#', Nth );				// break on Nth memory manager call
**		SetMemMgrBreak( '&', address);			// break when address is passed into the memory manager
**		SetMemMgrBreak( '&out', address);		// break when address is returned from the memory manager
**
*/
void SetMemMgrBreak ( long what, ... );


/*	----------------------------------------------------------------
**
**	CountHeapBlocks  --  Count blocks in a heap (of various types)
**
*/
void CountHeapBlocks
	(
		Size *		pTotalSize,			/* out: size of all matching blocks */
		ULong *		pFoundCount,		/* out: number of matching blocks */
		Heap		heap,				/* in: heap to visit (nil for current heap) */
		int			blockType,			/* in: block type filter (zero, kMM_HeapPtrBlock or kMM_HeapHandleBlock) */
		ULong		name,				/* in: name to match (or zero) */
		ULong		nameMask			/* in: mask of name (or zero) */
	);


/*	----------------------------------------------------------------
**
**	Heap Refcons and bases
**
*/
void *	GetHeapRefcon(Heap DEFAULT_NIL);			/* snarf refcon */
void	SetHeapRefcon(void *, Heap DEFAULT_NIL);	/* set refcon */
Heap	VoidStarToHeap(void *);						/* convert base addr to Heap */
void *	HeaptoVoidStar(Heap);						/* convert Heap to base addr */


#ifdef forDebug
/*	----------------------------------------------------------------
**
**	Heap operation tracing, heap tracking
**
*/
extern Boolean gMemTraceOn;

void MemTraceInit(ULong maxNum);
void MemTraceOn(Boolean);
ULong MemTraceCaptureStart(int op, ULong p1, ULong p2 DEFAULT_NIL);
void MemTraceCaptureEnd(ULong, int markNested);
void MemTraceDump(HammerIORef ref);
Heap RememberedHeapAt(int n);						/* return a heap we've (probably) remembered */


/*	----------------------------------------------------------------
**
**	Randomly return nil on allocations
**
*/
void SetRandomNilReturns
	(
		Heap	h,			/* heap to return nil on (nil for current heap) */
		ULong	freq,		/* frequency in 256-based percent zero --> never, 256 --> always */
		ULong	delay,		/* number of allocations to delay before turning on */
		ULong	duration	/* number of allocations to run before turning off */
	);
#endif /*forDebug*/

#ifdef __cplusplus
}
#endif


/*	----------------------------------------------------------------
**
**	Everyone should use the full names
**
*/
#define DisposeHandle DisposHandle
#define DisposePtr    DisposPtr

#endif /* __NEWTONMEMORY_H */
